package com.engine.core;

import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Paint;
import android.graphics.Rect;
import android.view.SurfaceHolder;

public class Graphics
{
    private final String mName = "Graphics";

    public SurfaceHolder mSurface = null;

    private Canvas mCanvas = null;

    private Paint mPaint = null;

    private int mWidth;
    private int mHeight;

    public Graphics() {}

    public int init()
    {
        Debug.inf(getClass().getName(), getName() + " initialized");
        mPaint = new Paint();

        return RetCode.SUCCESS;
    }

    public int term()
    {
        Debug.inf(getClass().getName(), getName() + " terminated");
        return RetCode.SUCCESS;
    }

    public String getName()
    {
        return mName;
    }

    public void setSurfaceHolder(SurfaceHolder surface)
    {
        mSurface = surface;
    }

    public void screenCreated(SurfaceHolder surface)
    {
        try
        {
            mSurface = surface;
            lock();
            mWidth = mCanvas.getWidth();
            mHeight = mCanvas.getHeight();
        }
        finally
        {
            unlock();
        }
    }

    public void screenDestroyed()
    {
    }

    public void screenChanged(int width, int height)
    {
        mWidth = width;
        mHeight = height;
    }

    public void lock()
    {
        mCanvas = mSurface.lockCanvas(null);
    }

    public void unlock()
    {
        if(mCanvas != null)
        {
            mSurface.unlockCanvasAndPost(mCanvas);
        }
    }

    public Canvas getCanvas()
    {
        return mCanvas;
    }

    public Paint getPaint()
    {
        return mPaint;
    }

    public int getWidth()
    {
        return mWidth;
    }

    public int getHeight()
    {
        return mHeight;
    }

    /**
     * Scales the bitmap from src to dst rectangle in the current canvas.
     * @param bitmap Bitmap to scale and draw.
     * @param src Source rectangle of the bitmap to be processed.
     * @param dst Destination rectangle of the bitmap to be processed.
     */
    public void drawBitmapScale(Bitmap bitmap, Rect src, Rect dst)
    {
        mCanvas.drawBitmap(bitmap, src, dst, mPaint);
    }

    /**
     * Draws the given bitmap in the given position.
     * @param bitmap Bitmap to draw.
     * @param x X position where bitmap should be drawn.
     * @param y Y position where bitmap should be drawn.
     */
    public void drawBitmap(Bitmap bitmap, float x, float y)
    {
        mCanvas.drawBitmap(bitmap, x, y, mPaint);
    }

    /**
     * Draws given rectangle on the current canvas.
     * @param rect
     */
    public void drawRect(Rect rect)
    {
        mCanvas.drawRect(rect, mPaint);
    }

    /**
     * Draws the text on the current canvas.
     * @param text Text to draw.
     * @param x X position of the text in the screen.
     * @param y Y position of the text in the screen.
     */
    public void drawText(String text, float x, float y)
    {
        mCanvas.drawText(text, x, y, mPaint);
    }

    /**
     * Draws the circle in the given position.
     * @param x X position of the circle center.
     * @param y Y position of the circle center.
     * @param radius Circle radius.
     * @param color Color of the circle.
     */
    public void drawCircle(float x, float y, float radius)
    {
        mCanvas.drawCircle(x, y, radius, mPaint);
    }
    
    public void setColor(int color)
    {
        mPaint.setColor(color);
    }
    
    public void setTextSize(int size)
    {
        mPaint.setTextSize(size);
    }
    
    public void setAlpha(int alpha)
    {
        mPaint.setAlpha(alpha);
    }
    
    public void setTextBold(boolean bold)
    {
        mPaint.setFakeBoldText(bold);
    }
    
    public int getTextSize(String text)
    {
    	int l = text.length();
    	float widths[] = new float[l];
        mPaint.getTextWidths(text, widths);
        float out = 0;
        for(int i = 0; i<l; i++)
        	out += widths[i];
        return (int)out;
    }
}
