<?php

namespace Tech\Http\Router;

use Tech\Http\Constants\Method;
use Tech\Http\Exception\HttpException;
use InvalidArgumentException;
use Tech\Collection\Collection;
use Tech\Http\Constants\Status;
use Tech\Http\Foundation\Request;

/**
 * @method self get(string $route, string $controller, string $action)
 * @method self head(string $route, string $controller, string $action)
 * @method self post(string $route, string $controller, string $action)
 * @method self put(string $route, string $controller, string $action)
 * @method self delete(string $route, string $controller, string $action)
 * @method self connect(string $route, string $controller, string $action)
 * @method self options(string $route, string $controller, string $action)
 * @method self trace(string $route, string $controller, string $action)
 * @method self patch(string $route, string $controller, string $action)
 */
class Router
{
    private Collection $routes;

    public function __construct()
    {
        $this->routes = new Collection();
    }

    public function __call($name, $arguments): self
    {
        $name = strtoupper($name);

        if (!in_array($name, Method::all())) {
            throw new InvalidArgumentException('Unsupported HTTP method provided.');
        }

        $this->routes[] = new Route($name, ...$arguments);

        return $this;
    }

    public function resolve(Request $request): Route
    {
        $partialMatches = $this->routes->where('route', '=', $request->getUri());

        if ($partialMatches->isEmpty()) {
            throw new HttpException(Status::NOT_FOUND, sprintf('No route found for %s.', $request->getUri()));
        }

        $match = $partialMatches->where('method', '=', $request->getMethod())->oneOrNull();

        if (!$match) {
            throw new HttpException(Status::METHOD_NOT_ALLOWED);
        }

        return $match;
    }
}