package com.engine.core;

import java.util.Enumeration;
import java.util.Vector;

public class TimerManager 
{
    private final String mName = "TimerManager";

    private Vector<Timer> mTimers = null;

    public TimerManager() {}

    public int init()
    {
        mTimers = new Vector<Timer>();

        Debug.inf(getClass().getName(), getName() + " initialized");
        return RetCode.SUCCESS;
    }

    public int term()
    {
        int ret = stopAll();
        if (ret != RetCode.SUCCESS)
        {
            Debug.err(getClass().getName(), "Problem with stopping timers!");
        }

        // Remove timers even if there were errors.
        mTimers.clear();

        Debug.inf(getClass().getName(), getName() + " terminated");
        return ret;
    }

    /**
     * Creates a new timer.
     * @return Timer.
     */
    public Timer create()
    {
        Timer timer = new Timer();
        mTimers.add(timer);

        return timer;
    }

    /**
     * Removes the timer with given id from the manager.
     * @param id
     * @return SUCCESS if timer removed, otherwise BAD_PARAM.
     */
    public int remove(int id)
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int i = 0;
        while (e.hasMoreElements())
        {
            timer = e.nextElement();
            if (timer.getId() == id)
            {
                Debug.inf(getClass().getName(), "Timer with id " + id + " found. Removing.");
                mTimers.remove(i);
                return RetCode.SUCCESS;
            }
            i++;
        }

        return RetCode.BAD_PARAM;
    }

    /**
     * Starts all the registered timers.
     * @return SUCCESS if timers started, otherwise FAILURE.
     */
    public int startAll()
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int ret = RetCode.SUCCESS;
        while(e.hasMoreElements())
        {
            timer = e.nextElement();
            ret = timer.start();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(), "Problem with starting timer (id: " + timer.getId());
                return ret;
            }
        }

        return ret;
    }

    /**
     * Stops all the registered timers.
     * @return SUCCESS if timers started, otherwise FAILURE.
     */
    public int stopAll()
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int ret = RetCode.SUCCESS;
        while (e.hasMoreElements())
        {
            timer = e.nextElement();
            ret = timer.stop();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(), "Problem with stopping timer (id: " + timer.getId());
                return ret;
            }
        }

        return ret;
    }

    /**
     * Pauses all the registered timers.
     * @return SUCCESS if timers started, otherwise FAILURE.
     */
    public int pauseAll()
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int ret = RetCode.SUCCESS;
        while (e.hasMoreElements())
        {
            timer = e.nextElement();
            ret = timer.pause();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(), "Problem with pausing timer (id: " + timer.getId());
                return ret;
            }
        }

        return ret;
    }

    /**
     * Resumes all the registered timers.
     * @return SUCCESS if timers resumed, otherwise FAILURE.
     */
    public int resumeAll()
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int ret = RetCode.SUCCESS;
        while (e.hasMoreElements())
        {
            timer = e.nextElement();
            ret = timer.resume();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(), "Problem with resuming timer (id: " + timer.getId());
                return ret;
            }
        }

        return ret;
    }

    /**
     * Updates all the registered timers.
     * @return SUCCESS if timers updated, otherwise FAILURE.
     */
    public int updateAll()
    {
        //Debug.inf(getClass().getName(), "updateAll");
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int ret = RetCode.SUCCESS;
        while(e.hasMoreElements())
        {
            timer = e.nextElement();
            ret = timer.update();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(), "Problem with updating timer (id: " + timer.getId());
                return ret;
            }
        }

        return ret;
    }

    /**
     * Stops the timer.
     * @param id Timer to be started.
     * @return SUCCESS if ok, otherwise FAILURE or BAD_PARAM.
     */
    public int start(int id)
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int ret = RetCode.SUCCESS;
        while (e.hasMoreElements())
        {
            timer = e.nextElement();
            if (timer.getId() == id)
            {
                Debug.inf(getClass().getName(), "Timer with id " + id + " found. Starting.");
                ret = timer.start();
                if (ret != RetCode.SUCCESS)
                {
                    Debug.err(getClass().getName(), "Problem with starting the timer!");
                }

                return ret;
            }
        }

        // We are here, so timer with given id doesn't exist.
        return RetCode.BAD_PARAM;
    }

    /**
     * Stop the timer.
     * @param id Timer to be stopped.
     * @return SUCCESS if ok, otherwise FAILURE or BAD_PARAM.
     */
    public int stop(int id)
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int ret = RetCode.SUCCESS;
        while (e.hasMoreElements())
        {
            timer = e.nextElement();
            if (timer.getId() == id)
            {
                Debug.inf(getClass().getName(), "Timer with id " + id + " found. Stopping.");
                ret = timer.stop();
                if (ret != RetCode.SUCCESS)
                {
                    Debug.err(getClass().getName(), "Problem with stopping the timer!");
                }

                return ret;
            }
        }

        // We are here, so timer with given id doesn't exist.
        return RetCode.BAD_PARAM;
    }

    /**
     * Pause the timer.
     * @param id Timer to be paused.
     * @return SUCCESS if ok, otherwise FAILURE or BAD_PARAM.
     */
    public int pause(int id)
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int ret = RetCode.SUCCESS;
        while (e.hasMoreElements())
        {
            timer = e.nextElement();
            if (timer.getId() == id)
            {
                Debug.inf(getClass().getName(), "Timer with id " + id + " found. Pausing.");
                ret = timer.pause();
                if (ret != RetCode.SUCCESS)
                {
                    Debug.err(getClass().getName(), "Problem with pausing the timer!");
                }

                return ret;
            }
        }

        // We are here, so timer with given id doesn't exist.
        return RetCode.BAD_PARAM;
    }

    /**
     * Resumes the timer.
     * @param id Timer to be resumed.
     * @return SUCCESS if ok, otherwise FAILURE or BAD_PARAM.
     */
    public int resume(int id)
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        int ret = RetCode.SUCCESS;
        while (e.hasMoreElements())
        {
            timer = e.nextElement();
            if (timer.getId() == id)
            {
                Debug.inf(getClass().getName(), "Timer with id " + id + " found. Resuming.");
                ret = timer.resume();
                if (ret != RetCode.SUCCESS)
                {
                    Debug.err(getClass().getName(), "Problem with resuming the timer!");
                }

                return ret;
            }
        }

        // We are here, so timer with given id doesn't exist.
        return RetCode.BAD_PARAM;
    }

    /**
     * Gets the timer with given id.
     * @param id Id of the timer.
     * @return Handle to the timer with given id or null if timer not found.
     */
    public Timer get(int id)
    {
        Timer timer = null;
        Enumeration<Timer> e = mTimers.elements();
        while (e.hasMoreElements())
        {
            timer = e.nextElement();
            if (timer.getId() == id)
            {
                Debug.inf(getClass().getName(), "Timer with id " + id + " found. Pausing.");
                return timer;
            }
        }

        // We are here, so timer with given id doesn't exist.
        return null;
    }

    /**
     * Returns the module name.
     * @return Module name.
     */
    public String getName()
    {
        return mName;
    }
}
