package com.engine;

import android.app.Activity;
import android.os.Bundle;
import android.view.Window;
import android.view.WindowManager;

import com.engine.core.Core;
import com.engine.core.Debug;
import com.engine.core.RetCode;

public class EngineActivity extends Activity 
{
    private static final String TAG = "EngineActivity";

    private EngineSurface mEngineView;
	
    /**
     * Determines if activity is going to be destroyed (false), 
     * or device is just rotating (true).
     */
    private boolean mWasSavingState = false;

	private int view = 0;
	private int id = 0;
	
    /** Called when the activity is first created. */
    public void init(int view, int id)
    {
    	this.view  = view;
    	this.id = id;
    }
    
    @Override
    public void onCreate(Bundle savedInstanceState)
    {
        super.onCreate(savedInstanceState);

        // Remove title bar.
        requestWindowFeature(Window.FEATURE_NO_TITLE);
        // Set the full screen mode.
        getWindow().setFlags(WindowManager.LayoutParams.FLAG_FULLSCREEN, WindowManager.LayoutParams.FLAG_FULLSCREEN); 
        
        setContentView(this.view);
        mEngineView = (EngineSurface)findViewById(this.id);
        
        if(savedInstanceState != null)
        {
            Debug.inf(TAG, "onCreate - after restoring the state!");
            mWasSavingState = false;

            // Update the surface holder.
            Core.getInstance().getGraphics().setSurfaceHolder(mEngineView.getHolder());
        }
        else
        {
            Debug.inf(TAG, "onCreate - first time, creating activity!");

            int ret = Core.getInstance().init(mEngineView.getHolder(), this);
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(TAG, "Problem with initializing the core!");
            }

            onInit();
        }
    }

    @Override
    public void onDestroy()
    {
        super.onDestroy();
        
        if(mWasSavingState == false)
        {
            Debug.inf(TAG, "onDestroy - activity finished!");
            int ret = Core.getInstance().stop();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(TAG, "Couldn't stop the core!");
            }

            ret = Core.getInstance().term();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(TAG, "Couldn't terminate the core!");
            }

            onTerm();
        }
        else
        {
            Debug.inf(TAG, "onDestroy - activity is just rotating?");
        }
    }

    /**
     * Called when core was initialized and screen was created. Called by the
     * core.
     */
    public void ready()
    {
        onReady();
    }

    @Override
    protected void onSaveInstanceState(Bundle outState)
    {
        super.onSaveInstanceState(outState);

        Debug.inf(TAG, "onSaveInstanceState()");

        mWasSavingState = true;

        // Stop the Core.
        Debug.inf(TAG, "Stopping the Core");
        int ret = Core.getInstance().stop();
        if (ret != RetCode.SUCCESS)
        {
            Debug.err(TAG, "Couldn't terminate the core!");
        }
    }

    /**
     * Called when activity is being initialized.
     */
    protected void onInit()
    {
        Debug.inf(TAG, "onInit");
    }

    /**
     * Called when activity is being terminated.
     */
    protected void onTerm()
    {
        Debug.inf(TAG, "onTerm");
    }

    /**
     * Called when core was initialized and the screen was created.
     */
    protected void onReady()
    {
        Debug.inf(TAG, "onReady");
    }

    @Override
    public void onPause()
    {
        super.onPause();
        Core.getInstance().pause();
        Debug.inf(TAG, "onPause");
    }

    @Override
    public void onResume()
    {
        super.onResume();
        Core.getInstance().resume();
        Debug.inf(TAG, "onResume");
    }
}
