package com.engine.core;

import java.util.Stack;

import com.engine.EngineActivity;

import android.app.Activity;
import android.content.Context;
import android.view.KeyEvent;
import android.view.MotionEvent;
import android.view.SurfaceHolder;

public class Core
{
    private static final String TAG = "Core";

    private class CoreThread extends Thread
    {
        /** Indicate whether the thread is running or not. */
        private boolean mRunning = false;

        private Core mCore = null;

        private int mSleepTime = 20;

        public CoreThread(Core core)
        {
            // core should never be null.
            this.mCore = core;
        }

        public int runThread()
        {
            setRunning(true);
            start();

            return RetCode.SUCCESS;
        }

        public int stopThread()
        {
            // Shut down the core thread.
            boolean retry = true;
            setRunning(false);
            while(retry)
            {
                try
                {
                    this.join();
                    retry = false;
                }
                catch (InterruptedException e)
                {
                    Debug.err(getClass().getName(), e.toString());
                    return RetCode.FAILURE;
                }
            }

            return RetCode.SUCCESS;
        }

        @Override
        public void run()
        {
            while(mRunning)
            {
                mCore.onDraw();
                // Call sleep to be able to get some inputs from the user.
                try
                {
                    sleep(mSleepTime);
                }
                catch(InterruptedException e)
                {
                    e.printStackTrace();
                }
            }
        }

        /**
         * Used to signal the thread whether it should be running or not.
         * Passing true allows the thread to run; passing false will shut it
         * down if it's already running. Calling start() after this was most
         * recently called with false will result in an immediate shutdown.
         * 
         * @param b
         *            true to run, false to shut down
         */
        public void setRunning(boolean run)
        {
            mRunning = run;
        }

        /**
         * Return the current thread running status. When tread beeing
         * terminated it may return FALSE, but thread is still running (code is
         * in the waiting state..).
         * 
         * @return Thread running status.
         */
    }

    private int mState = StateCode.TERMINATED;

    private TimerManager mTimerManager = null;

    private Graphics mRenderer = null;

    /** Context in which the core is running. */
    private Context mContext = null;

    private SurfaceHolder mSurface = null;

    /** Indicates if screen/canvas is ready for drawing. */
    private boolean mScreenExists = false;

    private CoreThread mCoreThread = null;

    private static Core mInstance = null;

    private static final Object mLock = new Object();

    private Stack<GameInterface> mGameStatesStack = null;

    private Timer mTimer;
    
    private boolean mFirstStart = true;

    /**
     * Core is a singleton.
     * 
     * @return Instance of the Core.
     */
    static synchronized public Core getInstance()
    {
        if (mInstance == null)
            mInstance = new Core();

        return mInstance;
    }

    private Core()
    {
        Debug.inf(getClass().getName(), "Core()");
        mTimer = new Timer();
        mGameStatesStack = new Stack<GameInterface>();
    }

    /**
     * Initializes the core.
     */
    public int init(SurfaceHolder surfaceHolder, Context context)
    {
        int ret = RetCode.SUCCESS;

        Debug.inf(getClass().getName(), "init()");

        if((mState == StateCode.INITIALIZED) || (mState == StateCode.RUNNING)
                || (mState == StateCode.STOPPED))
        {
            Debug.warn(getClass().getName(),
                    "Core already initialized, state: " + mState);
            return ret;
        }

        // We are here (STATE_TERMINATED), so go and initialize everything.

        if((surfaceHolder == null) || (context == null))
        {
            Debug.err(getClass().getName(), "Bad parameters!");
            return RetCode.BAD_PARAM;
        }

        mSurface = surfaceHolder;
        mContext = context;

        // mCoreThread = new CoreThread(this);

        // Initialize all the modules.
        mTimerManager = new TimerManager();
        ret = mTimerManager.init();
        if(ret != RetCode.SUCCESS)
        {
            Debug.err("Core", "Problem with initializing the TimerManager");
            return ret;
        }

        mRenderer = new Graphics();
        ret = mRenderer.init();
        if(ret != RetCode.SUCCESS)
        {
            Debug.err("Core", "Problem with initializing the Graphics");
            return ret;
        }

        Fps.reset();

        mRenderer.setSurfaceHolder(mSurface);

        mState = StateCode.INITIALIZED;

        mFirstStart = true;

        return ret;
    }

    /**
     * Terminates the core.
     */
    public int term()
    {
        int ret = RetCode.SUCCESS;

        Debug.inf(getClass().getName(), "term()");

        if(mState == StateCode.TERMINATED)
        {
            Debug.warn(getClass().getName(), "Core already terminated");
            // Return SUCCESS, as it doesn't have any negative impact.
            return ret;
        }

        if(mState == StateCode.RUNNING)
        {
            Debug.warn(getClass().getName(),
                    "Core is still running, stopping..");
            ret = stop();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(),
                        "Problem with stopping the core!");
                return ret;
            }
        }

        // We are in the proper state here so go and terminate everything.

        // Remove all active game states.
        ret = removeGameStates();
        if(ret != RetCode.SUCCESS)
        {
            Debug.err(getClass().getName(),
                    "Problem with removing the game states!");
        }
        mGameStatesStack.clear();

        // Don't really care about the return codes while terminating.
        // Just log error.
        ret = mTimerManager.term();
        if (ret != RetCode.SUCCESS)
        {
            Debug.err("Core", "Problem with terminating the TimerManager");
        }

        ret = mRenderer.term();
        if (ret != RetCode.SUCCESS)
        {
            Debug.err("Core", "Problem with terminating the Graphics");
        }

        mState = StateCode.TERMINATED;

        mFirstStart = true;

        return ret;
    }

    public int start()
    {
        Debug.inf(getClass().getName(), "start()");

        if(mState == StateCode.TERMINATED)
        {
            Debug.err(getClass().getName(), "Wrong core state: " + mState);
            return RetCode.FAILURE;
        }

        if(mState == StateCode.RUNNING)
        {
            Debug.warn(getClass().getName(), "Core already running.");
            return RetCode.SUCCESS;
        }

        mCoreThread = new CoreThread(this);

        int ret = mCoreThread.runThread();
        if(ret != RetCode.SUCCESS)
        {
            Debug.err(getClass().getName(),
                    "Problem with running the core thread!");
            return ret;
        }

        mState = StateCode.RUNNING;

        mTimer.start();

        if(mFirstStart)
        {
            Debug.inf(TAG, "Calling activity ready()");
            ((EngineActivity) mContext).ready();
        }

        return RetCode.SUCCESS;
    }

    public int stop()
    {
        Debug.inf(getClass().getName(), "stop()");

        if((mState == StateCode.INITIALIZED) || 
            (mState == StateCode.TERMINATED))
        {
            Debug.err(getClass().getName(), "Wrong core state: " + mState);
            return RetCode.FAILURE;
        }

        if(mState == StateCode.STOPPED)
        {
            Debug.warn(getClass().getName(), "Core already stopped.");
            return RetCode.SUCCESS;
        }

        int ret = mCoreThread.stopThread();
        if(ret != RetCode.SUCCESS)
        {
            Debug.err(getClass().getName(),
                    "Problem with stopping the core thread!");
            return ret;
        }

        mCoreThread = null;

        mState = StateCode.STOPPED;

        mTimer.stop();

        return RetCode.SUCCESS;
    }

    public int pause()
    {
        Debug.inf(getClass().getName(), "pause()");

        int ret = RetCode.SUCCESS;
        if (mGameStatesStack.isEmpty() == false)
        {
            // Pause the current state.
            ret = mGameStatesStack.peek().pause();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(),
                        "Problem with pausing the game state!");
                return ret;
            }
        }

        return RetCode.SUCCESS;
    }

    public int resume()
    {
        Debug.inf(getClass().getName(), "resume()");

        int ret = RetCode.SUCCESS;
        if(mGameStatesStack.isEmpty() == false)
        {
            // Pause the current state.
            ret = mGameStatesStack.peek().resume();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(),
                        "Problem with pausing the game state!");
                return ret;
            }
        }

        return RetCode.SUCCESS;
    }

    private long mAccumulator = 0;
    private final long TIME_STEP = 40;
    private final long MAX_ACCUMULATED_TIME = 1000;

    /**
     * Main loop of the core.
     */
	long t0 = System.currentTimeMillis();
	long t0fps = System.currentTimeMillis();
	float fDeltaTime = 0;
	float fTime;
	int cfps = 0;
	int nFPS = 0;
    protected int onDraw()
    {
        int ret = RetCode.SUCCESS;

        synchronized(mLock)
        {
            if(mScreenExists)
            {
                try
                {
                    mRenderer.lock();

                    mTimer.update();
                    mAccumulator += mTimer.getTimeDeltaMillis();

                    // Make sure that we don't update the game too
                    // many times.
                    if (mAccumulator > MAX_ACCUMULATED_TIME)
                        mAccumulator = MAX_ACCUMULATED_TIME;
                    else if (mAccumulator < 0)
                        mAccumulator = 0;

                    GameInterface currentState = null;
                    if(mGameStatesStack.isEmpty() == false)
                    {
                        currentState = mGameStatesStack.peek();
                        
                        while(mAccumulator>TIME_STEP)
                        {
                            // Update all active timers.
                            ret = mTimerManager.updateAll();
                            if(ret != RetCode.SUCCESS)
                            {
                                Debug.err(getClass().getName(),
                                        "Problem with updating timers!");
                                mRenderer.unlock();
                                return ret;
                            }
                            
                            // Update the current state logic.
                            ret = currentState.update();
                            if (ret != RetCode.SUCCESS)
                            {
                                Debug.err(getClass().getName(),
                                        "Problem with updating the state!");
                                mRenderer.unlock();
                                return ret;
                            }

                            mAccumulator -= TIME_STEP;
                        }
                        
                        ret = currentState.render();
                        if (ret != RetCode.SUCCESS)
                        {
                            Debug.err(getClass().getName(),
                                    "Problem with rendering the state!");
                            mRenderer.unlock();
                            return ret;
                        }
                    }
/*
                	int fps = 0;
                	long time = System.currentTimeMillis();
                	long dt;
                	float delta = 1000.0f/30.0f;
                    GameInterface currentState = null;
                    if(mGameStatesStack.isEmpty() == false)
                    {
            			do { dt=System.currentTimeMillis() - t0; } while(dt < 1);
            			
            			if(dt >= delta)
            			{
            				fDeltaTime=dt/1000.0f;
            				if(fDeltaTime > 0.2f)
            					fDeltaTime = delta !=0 ? delta/1000.0f : 0.01f;

            				fTime += fDeltaTime;

            				t0=System.currentTimeMillis();
            				if(t0-t0fps <= 1000) cfps++;
            				else
            				{
            					nFPS=cfps; cfps=0; t0fps=t0;
            				}
            				//else
            					//if(delta && dt+3 < delta) System.sleep(1);
                            currentState = mGameStatesStack.peek();

                            ret = currentState.render();
                            if (ret != RetCode.SUCCESS)
                            {
                                Debug.err(getClass().getName(),
                                        "Problem with rendering the state!");
                                mRenderer.unlock();
                                return ret;
                            }
            			}
                    }*/
                    Fps.update();
                }
                finally
                {
                    // An exception occured, but try to unlock the canvas.
                    mRenderer.unlock();
                }
            }
            else
            {
                Debug.inf(getClass().getName(),
                        "Screen not ready. Don't render.");
            }
        }
        return ret;
    }

    /**
     * Changes current game state. It replaces all active states with the new
     * one.
     */
    public int changeState(GameInterface gameState)
    {
        Debug.inf(getClass().getName(), "changeState()");

        if (gameState == null)
        {
            Debug.err(getClass().getName(), "Bad parameter!");
            return RetCode.BAD_PARAM;
        }

        // Remove / terminate all the states in the stack.
        int ret = RetCode.SUCCESS;

        ret = removeGameStates();
        if (ret != RetCode.SUCCESS)
        {
            Debug.err(getClass().getName(),
                    "Problem with removing the obsolete game states!");
            return ret;
        }

        mGameStatesStack.push(gameState);
        ret = mGameStatesStack.peek().init();
        if (ret != RetCode.SUCCESS)
        {
            Debug.err(getClass().getName(),
                    "Problem with initializing the game state!");
            return ret;
        }

        return ret;
    }

    /**
     * Pauses the current state and put the new state on the top.
     */
    public int pushState(GameInterface gameState)
    {
        Debug.inf(getClass().getName(), "pushState()");

        if (gameState == null)
        {
            Debug.err(getClass().getName(), "Bad parameter!");
            return RetCode.BAD_PARAM;
        }

        int ret = RetCode.SUCCESS;

        if (mGameStatesStack.isEmpty() == false)
        {
            // Pause the current state.
            ret = mGameStatesStack.peek().pause();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(),
                        "Problem with pausing the game state!");
                return ret;
            }
        }

        mGameStatesStack.push(gameState);
        ret = mGameStatesStack.peek().init();
        if (ret != RetCode.SUCCESS)
        {
            Debug.err(getClass().getName(),
                    "Problem with initializing the game state!");
            return ret;
        }

        return ret;
    }

    /**
     * Pops the current state and resumes the previous state.
     */
    public int popState()
    {
        Debug.inf(getClass().getName(), "popState()");

        int ret = RetCode.SUCCESS;

        if (mGameStatesStack.isEmpty() == false)
        {
            ret = mGameStatesStack.peek().term();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(),
                        "Problem with terminating the game stae!");
                return ret;
            }

            mGameStatesStack.pop();
        }

        // Resume the next state (if exists).
        if (mGameStatesStack.isEmpty() == false)
        {
            ret = mGameStatesStack.peek().resume();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(),
                        "Problem with resuming the game state!");
                return ret;
            }
        }

        return ret;
    }

    public boolean keyDown(int keyCode, KeyEvent msg)
    {
        synchronized (mLock)
        {
            if (mGameStatesStack.isEmpty() == false)
            {
                return mGameStatesStack.peek().onKeyDown(keyCode, msg);
            }
        }

        return false;
    }

    public boolean keyUp(int keyCode, KeyEvent msg)
    {
        synchronized(mLock)
        {
            if(mGameStatesStack.isEmpty() == false)
            {
                return mGameStatesStack.peek().onKeyUp(keyCode, msg);
            }
        }

        return false;
    }

    public boolean touchEvent(MotionEvent event)
    {
        synchronized (mLock)
        {
            if (mGameStatesStack.isEmpty() == false)
            {
                return mGameStatesStack.peek().onTouchEvent(event);
            }
            else
            {
                Debug.inf(getClass().getName(),
                        "Touch event, going to close app!");
                quit();
            }
        }

        return false;
    }

    public void quit()
    {
        // Core will be terminated from activity onDestroy method.
        Activity act = (Activity) mContext;
        act.finish();
    }

    public void focusChanged(boolean hasWindowFocus)
    {
        // TODO implement focusChanged.
    }

    public TimerManager getTimerManager()
    {
        return mTimerManager;
    }
    
    public Graphics getGraphics()
    {
        return mRenderer;
    }

    public Context getContext()
    {
        return mContext;
    }

    public void screenCreated(SurfaceHolder surface)
    {
        synchronized(mLock)
        {
            // Notify the renderer that screen created.
            mSurface = surface;
            mRenderer.screenCreated(surface);

            //if (mFirstStart)
            {
                // Screen created so we may start the rendering thread.
                Debug.inf(getClass().getName(), "Starting the thread");
                int ret = Core.getInstance().start();
                if (ret != RetCode.SUCCESS)
                {
                    Debug.err(getClass().getName(), "Couldn't start the core!");
                }
                mScreenExists = true;
                mFirstStart = false;
            }
            //else
            if (mFirstStart)
            {
                Debug.inf(getClass().getName(), "Screen created!");

                GameInterface currentState = null;
                if (mGameStatesStack.isEmpty() == false)
                {
                    currentState = mGameStatesStack.peek();
                    currentState.onScreenCreated();
                }
            }
        }

        Debug.inf(getClass().getName(), "screenCreated");
    }

    public void screenDestroyed(SurfaceHolder surface)
    {
        synchronized (mLock)
        {
            mRenderer.screenDestroyed();

            GameInterface currentState = null;
            if (mGameStatesStack.isEmpty() == false)
            {
                currentState = mGameStatesStack.peek();
                currentState.onScreenDestroyed();
            }
            mScreenExists = false;
        }

        Debug.inf(getClass().getName(), "screenDestroyed");
    }

    public void screenChanged(SurfaceHolder surface, int width, int height)
    {
        synchronized (mLock)
        {
            mRenderer.screenChanged(width, height);

            GameInterface currentState = null;
            if (mGameStatesStack.isEmpty() == false)
            {
                currentState = mGameStatesStack.peek();
                currentState.onScreenChanged(width, height);
            }
            mScreenExists = true;
        }
        Debug.inf(getClass().getName(), "screenChanged, width: " + width
                + ", height: " + height);
    }
    
    private int removeGameStates()
    {
        int ret = RetCode.SUCCESS;
        GameInterface tmpState = null;

        Debug.inf(getClass().getName(), "Removing game states");

        while(mGameStatesStack.isEmpty() == false)
        {
            tmpState = mGameStatesStack.pop();
            ret = tmpState.term();
            if (ret != RetCode.SUCCESS)
            {
                Debug.err(getClass().getName(),
                        "Problem with terminating the game state!");
                return ret;
            }
        }

        return ret;
    }
}
